import { createAdminClient } from "@/utils/supabase/admin";
import { NextResponse } from "next/server";

export async function POST(
  request: Request,
  { params }: { params: { id: string } }
) {
  try {
    const userId = params.id;
    const adminClient = createAdminClient();
    
    // 检查用户是否存在 - 使用auth API
    try {
      const { data: user, error } = await adminClient.auth.admin.getUserById(userId);
      
      if (error || !user) {
        console.error("找不到指定用户:", error);
        return NextResponse.json(
          { error: "找不到指定用户" },
          { status: 404 }
        );
      }
    } catch (userError) {
      console.error("查询用户时出错:", userError);
      return NextResponse.json(
        { error: "查询用户时出错" },
        { status: 500 }
      );
    }
    
    // 检查用户是否已经是管理员
    const { data: admin, error: adminError } = await adminClient
      .from('admin_users')
      .select('user_id')
      .eq('user_id', userId);
      
    if (adminError) {
      console.error("检查管理员状态失败:", adminError);
      return NextResponse.json(
        { error: "检查管理员状态失败" },
        { status: 500 }
      );
    }
    
    let result;
    
    // 如果已经是管理员，则移除管理员权限
    if (admin && admin.length > 0) {
      const { error: removeError } = await adminClient
        .from('admin_users')
        .delete()
        .eq('user_id', userId);
        
      if (removeError) {
        console.error("移除管理员权限失败:", removeError);
        return NextResponse.json(
          { error: "移除管理员权限失败" },
          { status: 500 }
        );
      }
      
      result = { isAdmin: false, message: "已成功移除管理员权限" };
    } 
    // 如果不是管理员，则添加管理员权限
    else {
      const { error: addError } = await adminClient
        .from('admin_users')
        .insert({ user_id: userId });
        
      if (addError) {
        console.error("添加管理员权限失败:", addError);
        return NextResponse.json(
          { error: "添加管理员权限失败" },
          { status: 500 }
        );
      }
      
      result = { isAdmin: true, message: "已成功添加管理员权限" };
    }
    
    return NextResponse.json(result);
  } catch (error) {
    console.error("切换管理员状态时出错:", error);
    return NextResponse.json(
      { error: "服务器内部错误" },
      { status: 500 }
    );
  }
} 